<?php
declare(strict_types=1);
@ini_set('memory_limit', '1024M');
@set_time_limit(0);
error_reporting(E_ALL);
session_start();

function find_wp_load(): ?string {
    $candidates = [
        __DIR__ . '/wp-load.php',
        __DIR__ . '/../wp-load.php',
        __DIR__ . '/public_html/wp-load.php',
        dirname(__DIR__) . '/wp-load.php',
    ];
    foreach ($candidates as $path) if (is_file($path)) return $path;
    return null;
}
$wp_load = find_wp_load();
if (!$wp_load) {
    http_response_code(500);
    echo "<h1>Error</h1><p>No se encontró <code>wp-load.php</code>. Mueve este archivo a la raíz del WordPress.</p>";
    exit;
}
require_once $wp_load;

if (!is_user_logged_in() || !current_user_can('manage_options')) {
    auth_redirect();
    exit;
}
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];

function h(string $s): string { return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }
function normalize_heading_text(string $s): string {
    if (!function_exists('mb_strtolower')) return strtolower($s);
    $s = html_entity_decode($s, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    $s = strip_tags($s);
    $s = preg_replace('/\s+/u', ' ', $s ?? '');
    $s = trim((string)$s);
    $s = mb_strtolower($s, 'UTF-8');
    return $s;
}
function extract_headings(string $html, array $levels): array {
    if (!class_exists('DOMDocument')) return [];
    $out = [];
    if (trim($html) === '') return $out;
    $dom = new DOMDocument('1.0', 'UTF-8');
    libxml_use_internal_errors(true);
    $dom->loadHTML('<meta charset="UTF-8"/><div data-w="1">'.$html.'</div>', LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
    libxml_clear_errors();
    $xp = new DOMXPath($dom);
    $tags = array_map(fn($n)=>'h'.$n, $levels);
    $query = './/'.implode(' | .//', $tags);
    $nodes = $xp->query($query);
    if (!$nodes) return $out;
    foreach ($nodes as $n) {
        $level = (int)substr($n->nodeName, 1);
        $text  = trim($n->textContent ?? '');
        $norm  = normalize_heading_text($text);
        if ($norm !== '') {
            $out[] = ['level'=>$level, 'text'=>$text, 'norm'=>$norm, 'node'=>$n];
        }
    }
    return $out;
}
function rewrite_selected_headings(string $html, array $levels, string $toTag, array $allowedNorms, bool $preserveAttrs=true, bool $a11y=true): array {
    if (!class_exists('DOMDocument')) return [$html, 0];
    $changes = 0;
    if ($html === '' || empty($allowedNorms)) return [$html, 0];
    $dom = new DOMDocument('1.0', 'UTF-8');
    libxml_use_internal_errors(true);
    $dom->loadHTML('<meta charset="UTF-8"/><div data-w="1">'.$html.'</div>', LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
    libxml_clear_errors();
    $xp = new DOMXPath($dom);
    $tags = array_map(fn($n)=>'h'.$n, $levels);
    $query = './/'.implode(' | .//', $tags);
    $nodes = $xp->query($query);
    if (!$nodes || $nodes->length === 0) {
        $out = $dom->saveHTML($dom->getElementsByTagName('div')->item(0));
        $out = preg_replace('#^<div[^>]*data-w="1"[^>]*>|</div>$#', '', $out ?? '');
        return [$out, 0];
    }
    $toTag = strtolower($toTag);
    $isHeadingTarget = in_array($toTag, ['h1','h2','h3','h4','h5','h6'], true);
    $snapshot = [];
    foreach ($nodes as $n) $snapshot[] = $n;
    foreach ($snapshot as $node) {
        $text = trim($node->textContent ?? '');
        $norm = normalize_heading_text($text);
        if (!isset($allowedNorms[$norm])) continue;
        $origLevel = (int)substr($node->nodeName, 1);
        $new = $dom->createElement($toTag);
        if ($preserveAttrs && $node->hasAttributes()) {
            foreach (iterator_to_array($node->attributes) as $attr) {
                $new->setAttribute($attr->nodeName, $attr->nodeValue);
            }
        }
        if (!$isHeadingTarget && $a11y) {
            if (!$new->hasAttribute('role')) $new->setAttribute('role','heading');
            if (!$new->hasAttribute('aria-level')) $new->setAttribute('aria-level', (string)$origLevel);
        }
        while ($node->firstChild) $new->appendChild($node->firstChild);
        $node->parentNode->replaceChild($new, $node);
        $changes++;
    }
    $wrapper = $dom->getElementsByTagName('div')->item(0);
    $out = $dom->saveHTML($wrapper);
    $out = preg_replace('#^<div[^>]*data-w="1"[^>]*>|</div>$#', '', $out ?? '');
    return [$out, $changes];
}

$defaults = [
    'mode'            => 'scan',
    'from_levels'     => ['2'],
    'to_tag'          => 'span',
    'post_types'      => ['post','page'],
    'post_statuses'   => ['publish'],
    'limit'           => '1000',
    'offset'          => '0',
    'batch_size'      => '200',
    'dry_run'         => '1',
    'backup'          => '1',
];
$data = $defaults;
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!hash_equals($csrf, $_POST['csrf'] ?? '')) wp_die('CSRF inválido.');
    $data['mode']          = in_array($_POST['mode'] ?? 'scan', ['scan','apply'], true) ? $_POST['mode'] : 'scan';
    $levels                 = (array)($_POST['from_levels'] ?? ['2']);
    $levels                 = array_values(array_intersect($levels, ['1','2','3','4','5','6']));
    $data['from_levels']    = !empty($levels) ? $levels : ['2'];
    $data['to_tag']         = strtolower(trim($_POST['to_tag'] ?? 'span'));
    $data['post_types']     = array_map('sanitize_text_field', (array)($_POST['post_types'] ?? ['post','page']));
    $data['post_statuses']  = array_map('sanitize_text_field', (array)($_POST['post_statuses'] ?? ['publish']));
    $data['limit']          = (string) max(0, (int)($_POST['limit'] ?? 1000));
    $data['offset']         = (string) max(0, (int)($_POST['offset'] ?? 0));
    $data['batch_size']     = (string) max(1, (int)($_POST['batch_size'] ?? 200));
    $data['dry_run']        = !empty($_POST['dry_run']) ? '1' : '0';
    $data['backup']         = !empty($_POST['backup']) ? '1' : '0';
}

$scan_summary = null;
$duplicates = [];
if ($data['mode'] === 'scan' || $data['mode'] === 'apply') {
    $limit     = (int)$data['limit'];
    $offset    = (int)$data['offset'];
    $batchSize = (int)$data['batch_size'];
    $levelsInt = array_map('intval', $data['from_levels']);
    $totalProcessed = 0;
    $remaining = $limit > 0 ? $limit : PHP_INT_MAX;
    while ($remaining > 0) {
        $ppp = min($batchSize, $remaining);
        $q = new WP_Query([
            'post_type'      => $data['post_types'],
            'post_status'    => $data['post_statuses'],
            'posts_per_page' => $ppp,
            'offset'         => $offset,
            'orderby'        => 'ID',
            'order'          => 'ASC',
            'fields'         => 'all',
            'no_found_rows'  => true,
        ]);
        if (!$q->have_posts()) break;
        while ($q->have_posts()) {
            $q->the_post();
            $post = get_post();
            if (!$post) continue;
            $totalProcessed++;
            $found = extract_headings((string)$post->post_content, $levelsInt);
            if (empty($found)) continue;
            $url = get_permalink($post);
            $title = get_the_title($post);
            foreach ($found as $h) {
                $norm = $h['norm'];
                $level = (int)$h['level'];
                if (!isset($duplicates[$norm])) {
                    $duplicates[$norm] = ['display' => $h['text'], 'count' => 0, 'levels' => [], 'posts' => []];
                }
                $duplicates[$norm]['count']++;
                if (!in_array($level, $duplicates[$norm]['levels'], true)) {
                    $duplicates[$norm]['levels'][] = $level;
                }
                $keyPost = $post->ID.'#'.$norm;
                if (!isset($duplicates[$norm]['__seen'])) $duplicates[$norm]['__seen'] = [];
                if (!isset($duplicates[$norm]['__seen'][$keyPost])) {
                    $duplicates[$norm]['posts'][] = ['id'=>$post->ID, 'title'=>$title, 'url'=>$url];
                    $duplicates[$norm]['__seen'][$keyPost] = true;
                }
            }
        }
        wp_reset_postdata();
        $offset += $ppp;
        $remaining -= $ppp;
    }
    foreach ($duplicates as $k => &$v) {
        unset($v['__seen']);
        if (count($v['posts']) < 2 && $v['count'] < 2) {
            unset($duplicates[$k]);
            continue;
        }
        sort($v['levels']);
    } unset($v);
    uasort($duplicates, function($a, $b) {
        $pa = count($a['posts']); $pb = count($b['posts']);
        if ($pa !== $pb) return $pb <=> $pa;
        return $b['count'] <=> $a['count'];
    });
    $scan_summary = sprintf('Escaneados %d posts. Grupos de encabezados repetidos encontrados: %d', $totalProcessed, count($duplicates));
}

$apply_notice = '';
if ($data['mode'] === 'apply' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $selected = (array)($_POST['selected_norms'] ?? []);
    $selected = array_unique(array_map('strval', $selected));
    $selectedMap = array_fill_keys($selected, true);
    if (!empty($selectedMap)) {
        $limit     = (int)$data['limit'];
        $offset    = (int)$data['offset'];
        $batchSize = (int)$data['batch_size'];
        $levelsInt = array_map('intval', $data['from_levels']);
        $isDry     = $data['dry_run'] === '1';
        $doBackup  = $data['backup'] === '1';
        $totalProcessed = 0;
        $totalChanged   = 0;
        $postsTouched   = 0;
        $remaining = $limit > 0 ? $limit : PHP_INT_MAX;
        while ($remaining > 0) {
            $ppp = min($batchSize, $remaining);
            $q = new WP_Query([
                'post_type'      => $data['post_types'],
                'post_status'    => $data['post_statuses'],
                'posts_per_page' => $ppp,
                'offset'         => $offset,
                'orderby'        => 'ID',
                'order'          => 'ASC',
                'fields'         => 'all',
                'no_found_rows'  => true,
            ]);
            if (!$q->have_posts()) break;
            while ($q->have_posts()) {
                $q->the_post();
                $post = get_post();
                if (!$post) continue;
                $totalProcessed++;
                [$newHtml, $changes] = rewrite_selected_headings((string)$post->post_content, $levelsInt, $data['to_tag'], $selectedMap, true, true);
                if ($changes > 0 && $newHtml !== (string)$post->post_content) {
                    $totalChanged += $changes;
                    $postsTouched++;
                    if (!$isDry) {
                        if ($doBackup) {
                            add_post_meta($post->ID, '_heading_rewriter_backup', ['time' => current_time('mysql'), 'from_lv' => $data['from_levels'], 'to' => $data['to_tag'], 'content' => (string)$post->post_content], false);
                        }
                        wp_update_post(['ID' => $post->ID, 'post_content' => $newHtml]);
                    }
                }
            }
            wp_reset_postdata();
            $offset += $ppp;
            $remaining -= $ppp;
        }
        $apply_notice = sprintf('Aplicación terminada. Posts procesados: %d | Posts modificados: %d | Encabezados cambiados: %d%s', $totalProcessed, $postsTouched, $totalChanged, ($data['dry_run']==='1' ? ' (simulación, no se guardó nada)' : ''));
    } else {
        $apply_notice = 'No seleccionaste ningún grupo de encabezados para cambiar.';
    }
}

$allPostTypes = get_post_types(['public' => true], 'names');
$allStatuses  = ['publish','draft','pending','future','private'];
?>
<!doctype html>
<html lang="es">
<head>
<meta charset="utf-8">
<title>Duplicados de encabezados y reescritura selectiva</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<style>
body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Ubuntu,Inter,Arial,sans-serif;background:#0b0f17;color:#e6edf3;margin:0}
.container{max-width:1200px;margin:40px auto;padding:0 20px}
.card{background:#111827;border:1px solid #1f2937;border-radius:14px;padding:20px;box-shadow:0 4px 24px rgba(0,0,0,.25)}
h1{font-size:20px;margin:0 0 14px}
.grid{display:grid;grid-template-columns:repeat(4,1fr);gap:12px}
.grid .col-span-2{grid-column:span 2}
label{display:block;font-size:12px;color:#9ca3af;margin-bottom:6px}
input[type=text],select{width:100%;padding:10px;border-radius:10px;border:1px solid #374151;background:#0f172a;color:#e6edf3}
.checkbox{display:flex;gap:10px;align-items:center;margin:6px 0}
.btn{display:inline-block;padding:10px 16px;border-radius:10px;border:1px solid #374151;background:#1f2937;color:#e6edf3;text-decoration:none;cursor:pointer}
.btn.primary{background:#2563eb;border-color:#1d4ed8}
.note{margin:14px 0;padding:10px;border-radius:10px;background:#0f172a;border:1px solid #1f2937;color:#cbd5e1}
.table{width:100%;border-collapse:collapse;margin-top:16px}
.table th,.table td{border-bottom:1px solid #243244;padding:8px 6px;font-size:13px;text-align:left;vertical-align:top}
.badge{display:inline-block;padding:2px 8px;border:1px solid #374151;border-radius:999px;font-size:12px;color:#cbd5e1}
.small{font-size:12px;color:#9ca3af}
.urls a{color:#93c5fd;text-decoration:none}
.sticky-actions{position:sticky;bottom:0;background:#0b0f17;padding:10px 0;margin-top:10px;border-top:1px solid #1f2937}
.credit{margin-top:16px;padding-top:10px;border-top:1px solid #1f2937;font-size:12px;color:#9ca3af}
.credit a{color:#93c5fd;text-decoration:none}
</style>
</head>
<body>
<div class="container">
  <div class="card">
    <h1>Duplicados de encabezados y reescritura selectiva</h1>

    <?php if ($scan_summary): ?>
      <div class="note"><?=h($scan_summary)?></div>
    <?php else: ?>
      <div class="note">Primero ejecuta un <b>escaneo</b> para detectar encabezados repetidos por niveles (H1–H6).</div>
    <?php endif; ?>

    <?php if (!empty($apply_notice)): ?>
      <div class="note"><b>Resultado:</b> <?=h($apply_notice)?></div>
    <?php endif; ?>

    <form method="post">
      <input type="hidden" name="csrf" value="<?=h($csrf)?>">
      <div class="grid">
        <div class="col-span-2">
          <label>Niveles origen (H*) a analizar / reescribir</label>
          <select name="from_levels[]" multiple size="6">
            <?php foreach (['1','2','3','4','5','6'] as $lv): ?>
              <option value="<?=$lv?>" <?= in_array($lv, $data['from_levels'], true)?'selected':''; ?>><?= 'H'.$lv ?></option>
            <?php endforeach; ?>
          </select>
          <div class="small">Mantén Ctrl/Cmd para seleccionar varios.</div>
        </div>

        <div>
          <label>Tag destino (al aplicar)</label>
          <select name="to_tag">
            <?php foreach (['span','div','p','h1','h2','h3','h4','h5','h6'] as $opt): ?>
              <option value="<?=$opt?>" <?= $data['to_tag']===$opt?'selected':''; ?>><?= strtoupper($opt) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label>Tamaño de lote</label>
          <input type="text" name="batch_size" value="<?=h($data['batch_size'])?>">
        </div>

        <div>
          <label>Límite total a procesar</label>
          <input type="text" name="limit" value="<?=h($data['limit'])?>">
        </div>

        <div>
          <label>Offset inicial</label>
          <input type="text" name="offset" value="<?=h($data['offset'])?>">
        </div>

        <div class="col-span-2">
          <label>Tipos de post</label>
          <select name="post_types[]" multiple size="4">
            <?php foreach ($allPostTypes as $pt): ?>
              <option value="<?=h($pt)?>" <?= in_array($pt, $data['post_types'], true)?'selected':''; ?>><?=h($pt)?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="col-span-2">
          <label>Estados</label>
          <select name="post_statuses[]" multiple size="5">
            <?php foreach ($allStatuses as $st): ?>
              <option value="<?=h($st)?>" <?= in_array($st, $data['post_statuses'], true)?'selected':''; ?>><?=h($st)?></option>
            <?php endforeach; ?>
          </select>
        </div>
      </div>

      <div class="grid" style="margin-top:8px">
        <div class="checkbox">
          <input type="checkbox" id="dry_run" name="dry_run" value="1" <?= $data['dry_run']==='1'?'checked':''; ?>>
          <label for="dry_run"><b>Simulación (dry-run)</b> — lista y aplica sin guardar</label>
        </div>
        <div class="checkbox">
          <input type="checkbox" id="backup" name="backup" value="1" <?= $data['backup']==='1'?'checked':''; ?>>
          <label for="backup">Guardar copia de seguridad en postmeta</label>
        </div>
      </div>

      <div style="margin-top:10px">
        <button class="btn" type="submit" name="mode" value="scan">1) Escanear duplicados</button>
      </div>

      <?php if (!empty($duplicates)): ?>
        <h2 style="margin-top:22px;font-size:16px">Grupos de encabezados iguales encontrados</h2>
        <table class="table">
          <thead>
            <tr>
              <th style="width:28px"><input type="checkbox" onclick="document.querySelectorAll('.chk-norm').forEach(cb=>cb.checked=this.checked)"></th>
              <th>Encabezado (texto)</th>
              <th>Niveles</th>
              <th>Veces</th>
              <th>Ejemplos de URLs</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($duplicates as $norm => $row): ?>
            <tr>
              <td><input class="chk-norm" type="checkbox" name="selected_norms[]" value="<?=h($norm)?>"></td>
              <td><?= h($row['display']) ?></td>
              <td>
                <?php foreach ($row['levels'] as $lv): ?>
                  <span class="badge"><?= 'H'.$lv ?></span>
                <?php endforeach; ?>
              </td>
              <td><span class="badge"><?= (int)$row['count'] ?></span></td>
              <td class="urls">
                <?php
                  $examples = array_slice($row['posts'], 0, 5);
                  foreach ($examples as $ex) {
                      echo '<div><a href="'.esc_url($ex['url']).'" target="_blank" rel="noopener">'.h($ex['title']).'</a></div>';
                  }
                  if (count($row['posts']) > 5) {
                      echo '<div class="small">… y '.(count($row['posts'])-5).' más</div>';
                  }
                ?>
              </td>
            </tr>
            <?php endforeach; ?>
          </tbody>
        </table>

        <div class="sticky-actions">
          <button class="btn primary" type="submit" name="mode" value="apply">2) Aplicar cambios a los seleccionados</button>
        </div>
      <?php endif; ?>
    </form>

    <div class="credit">
      Creado por <a href="https://victormisa.com" target="_blank" rel="noopener">Víctor Misa</a>
    </div>
  </div>
</div>
</body>
</html>
